#!/usr/bin/env python3
"""
Patch #2 for spy_trader.py
Fixes:
1. P&L calculation - tracks actual realized profit/loss, not gross cash flow
2. Daily loss check timing - skip_loss_check flag for sell+rebuy sequences
3. Order confirmation - polls Robinhood for fill status
4. ntfy logging - all notifications saved to ntfy_log.txt

Run: python3 patch_fixes.py
"""

import re

filepath = "/home/mm/moltbot-research/spy_trader.py"

with open(filepath, "r") as f:
    content = f.read()

# =============================================
# FIX 4: ntfy logging - modify notify() function
# =============================================
old_notify = '''def notify(message):
    """Send push notification via ntfy"""
    try:
        subprocess.run(
            ["curl", "-s", "-d", message, f"ntfy.sh/{NTFY_TOPIC}"],
            capture_output=True, timeout=10
        )
    except Exception as e:
        print(f"Notification failed: {e}")'''

new_notify = '''NTFY_LOG = os.path.join(WORKSPACE, "ntfy_log.txt")

def notify(message):
    """Send push notification via ntfy and log locally"""
    # Log to file first (always, even if send fails)
    try:
        with open(NTFY_LOG, "a") as f:
            f.write(f"[{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}] {message}\\n")
    except Exception as e:
        print(f"ntfy log write failed: {e}")

    # Send notification
    try:
        subprocess.run(
            ["curl", "-s", "-d", message, f"ntfy.sh/{NTFY_TOPIC}"],
            capture_output=True, timeout=10
        )
    except Exception as e:
        print(f"Notification failed: {e}")'''

content = content.replace(old_notify, new_notify)

# =============================================
# FIX 3: Add order confirmation polling
# =============================================
# Add time import if not present
if "import time" not in content:
    content = content.replace(
        "from datetime import datetime, date",
        "import time\nfrom datetime import datetime, date"
    )

# Add poll function after logout()
poll_func = '''
def poll_order_status(order_id, max_wait=15):
    """Poll Robinhood for order fill status"""
    if not order_id:
        return "unknown"
    for i in range(max_wait):
        try:
            info = rh.orders.get_stock_order_info(order_id)
            state = info.get("state", "unknown")
            if state in ("filled", "cancelled", "failed", "rejected"):
                return state
            time.sleep(1)
        except Exception:
            time.sleep(1)
    return "timeout (may still fill)"

'''

content = content.replace(
    '# ============================================\n# TRADE LOGGING',
    poll_func + '# ============================================\n# TRADE LOGGING'
)

# =============================================
# FIX 1: P&L calculation - add realized P&L tracking
# =============================================
# Add realized P&L file path
content = content.replace(
    'DAILY_LOG = os.path.join(WORKSPACE, "daily_pnl.csv")',
    'DAILY_LOG = os.path.join(WORKSPACE, "daily_pnl.csv")\nREALIZED_PNL_LOG = os.path.join(WORKSPACE, "realized_pnl.csv")'
)

# Add function to log realized P&L
realized_pnl_func = '''
def log_realized_pnl(sell_price, buy_price, shares, ticker):
    """Log actual realized profit/loss on a closed position"""
    pnl = (sell_price - buy_price) * shares
    header = "timestamp,ticker,shares,buy_price,sell_price,realized_pnl\\n"
    if not os.path.exists(REALIZED_PNL_LOG):
        with open(REALIZED_PNL_LOG, "w") as f:
            f.write(header)
    with open(REALIZED_PNL_LOG, "a") as f:
        f.write(f"{datetime.now().isoformat()},{ticker},{shares:.4f},"
                f"{buy_price:.2f},{sell_price:.2f},{pnl:.2f}\\n")
    return pnl

'''

content = content.replace(
    'def get_todays_trades():',
    realized_pnl_func + 'def get_todays_trades():'
)

# Replace get_todays_pnl with version that reads realized_pnl.csv
old_pnl = '''def get_todays_pnl():
    """Calculate today's realized P&L from trade log"""
    if not os.path.exists(TRADE_LOG):
        return 0.0
    today = date.today().isoformat()
    pnl = 0.0
    with open(TRADE_LOG, "r") as f:
        reader = csv.DictReader(f)
        for row in reader:
            if row["timestamp"].startswith(today) and row["status"] == "executed":
                total = float(row["total"])
                if row["action"] == "SELL":
                    pnl += total
                elif row["action"] == "BUY":
                    pnl -= total
    return pnl'''

new_pnl = '''def get_todays_pnl():
    """Calculate today's ACTUAL realized P&L from closed positions"""
    if not os.path.exists(REALIZED_PNL_LOG):
        return 0.0
    today = date.today().isoformat()
    pnl = 0.0
    with open(REALIZED_PNL_LOG, "r") as f:
        reader = csv.DictReader(f)
        for row in reader:
            if row["timestamp"].startswith(today):
                pnl += float(row["realized_pnl"])
    return pnl'''

content = content.replace(old_pnl, new_pnl)

# =============================================
# FIX 2 + 3: Update buy_spy with skip_loss_check and order polling
# =============================================
old_buy = '''def buy_spy(shares, reason="Signal"):
    """Execute buy order with risk checks"""
    price = float(rh.stocks.get_latest_price(TICKER)[0])
    can, msg = can_buy(price, shares)

    if not can:
        print(f"BUY BLOCKED: {msg}")
        log_trade("BUY", TICKER, shares, price, reason, f"blocked: {msg}")
        notify(f"BUY BLOCKED: {shares} {TICKER} @ ${price:.2f} - {msg}")
        return False

    try:
        order = rh.orders.order_buy_market(TICKER, shares)
        status = order.get("state", "unknown")
        print(f"BUY ORDER: {shares} {TICKER} @ ~${price:.2f} - Status: {status}")
        log_trade("BUY", TICKER, shares, price, reason, "executed")
        notify(f"BUY EXECUTED: {shares} {TICKER} @ ${price:.2f} - {reason}")
        update_positions()
        return True
    except Exception as e:
        print(f"BUY FAILED: {e}")
        log_trade("BUY", TICKER, shares, price, reason, f"failed: {e}")
        notify(f"BUY FAILED: {shares} {TICKER} - {e}")
        return False'''

new_buy = '''def buy_spy(shares, reason="Signal", skip_loss_check=False):
    """Execute buy order with risk checks"""
    price = float(rh.stocks.get_latest_price(TICKER)[0])
    can, msg = can_buy(price, shares, skip_loss_check=skip_loss_check)

    if not can:
        print(f"BUY BLOCKED: {msg}")
        log_trade("BUY", TICKER, shares, price, reason, f"blocked: {msg}")
        notify(f"BUY BLOCKED: {shares} {TICKER} @ ${price:.2f} - {msg}")
        return False

    try:
        order = rh.orders.order_buy_market(TICKER, shares)
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        print(f"BUY ORDER: {shares} {TICKER} @ ~${price:.2f} - Status: {status}")
        log_trade("BUY", TICKER, shares, price, reason, status)
        notify(f"BUY {status.upper()}: {shares} {TICKER} @ ${price:.2f} - {reason}")
        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"BUY FAILED: {e}")
        log_trade("BUY", TICKER, shares, price, reason, f"failed: {e}")
        notify(f"BUY FAILED: {shares} {TICKER} - {e}")
        return False'''

content = content.replace(old_buy, new_buy)

# Update can_buy to accept skip_loss_check
old_can_buy = '''def can_buy(price, shares):
    """Check all risk controls before buying"""
    reasons = []

    # Halt check
    if check_halt():
        return False, "HALTED"

    # Daily trade limit
    if get_todays_trades() >= MAX_DAILY_TRADES:
        reasons.append(f"Daily trade limit reached ({MAX_DAILY_TRADES})")

    # Daily loss limit
    daily_pnl = get_todays_pnl()
    if daily_pnl <= -MAX_DAILY_LOSS:
        reasons.append(f"Daily loss limit hit (${daily_pnl:.2f})")'''

new_can_buy = '''def can_buy(price, shares, skip_loss_check=False):
    """Check all risk controls before buying"""
    reasons = []

    # Halt check
    if check_halt():
        return False, "HALTED"

    # Daily trade limit
    if get_todays_trades() >= MAX_DAILY_TRADES:
        reasons.append(f"Daily trade limit reached ({MAX_DAILY_TRADES})")

    # Daily loss limit (skip when doing sell+rebuy sequence)
    if not skip_loss_check:
        daily_pnl = get_todays_pnl()
        if daily_pnl <= -MAX_DAILY_LOSS:
            reasons.append(f"Daily loss limit hit (${daily_pnl:.2f})")'''

content = content.replace(old_can_buy, new_can_buy)

# Update can_buy_dollars similarly
old_can_buy_d = '''def can_buy_dollars(amount):
    """Check all risk controls before buying by dollar amount"""
    reasons = []

    if check_halt():
        return False, "HALTED"

    if get_todays_trades() >= MAX_DAILY_TRADES:
        reasons.append(f"Daily trade limit reached ({MAX_DAILY_TRADES})")

    daily_pnl = get_todays_pnl()
    if daily_pnl <= -MAX_DAILY_LOSS:
        reasons.append(f"Daily loss limit hit (${daily_pnl:.2f})")'''

new_can_buy_d = '''def can_buy_dollars(amount, skip_loss_check=False):
    """Check all risk controls before buying by dollar amount"""
    reasons = []

    if check_halt():
        return False, "HALTED"

    if get_todays_trades() >= MAX_DAILY_TRADES:
        reasons.append(f"Daily trade limit reached ({MAX_DAILY_TRADES})")

    if not skip_loss_check:
        daily_pnl = get_todays_pnl()
        if daily_pnl <= -MAX_DAILY_LOSS:
            reasons.append(f"Daily loss limit hit (${daily_pnl:.2f})")'''

content = content.replace(old_can_buy_d, new_can_buy_d)

# Update buy_spy_dollars with skip_loss_check and order polling
old_buy_d = '''def buy_spy_dollars(amount, reason="Signal"):
    """Execute buy order by dollar amount (fractional shares)"""
    can, msg = can_buy_dollars(amount)

    if not can:
        print(f"BUY BLOCKED: {msg}")
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"blocked: {msg}")
        notify(f"BUY BLOCKED: ${amount} {TICKER} - {msg}")
        return False

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_buy_fractional_by_price(TICKER, amount, timeInForce='gfd')
        status = order.get("state", "unknown")
        est_shares = round(amount / price, 4)
        print(f"BUY ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("BUY", TICKER, est_shares, price, reason, "executed")
        notify(f"BUY EXECUTED: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")
        update_positions()
        return True
    except Exception as e:
        print(f"BUY FAILED: {e}")
        price = float(rh.stocks.get_latest_price(TICKER)[0]) if True else 0
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"BUY FAILED: ${amount} {TICKER} - {e}")
        return False'''

new_buy_d = '''def buy_spy_dollars(amount, reason="Signal", skip_loss_check=False):
    """Execute buy order by dollar amount (fractional shares)"""
    can, msg = can_buy_dollars(amount, skip_loss_check=skip_loss_check)

    if not can:
        print(f"BUY BLOCKED: {msg}")
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"blocked: {msg}")
        notify(f"BUY BLOCKED: ${amount} {TICKER} - {msg}")
        return False

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_buy_fractional_by_price(TICKER, amount, timeInForce='gfd')
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        est_shares = round(amount / price, 4)
        print(f"BUY ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("BUY", TICKER, est_shares, price, reason, status)
        notify(f"BUY {status.upper()}: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")
        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"BUY FAILED: {e}")
        try:
            price = float(rh.stocks.get_latest_price(TICKER)[0])
        except:
            price = 0
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"BUY FAILED: ${amount} {TICKER} - {e}")
        return False'''

content = content.replace(old_buy_d, new_buy_d)

# Update sell_spy with realized P&L logging and order polling
old_sell = '''def sell_spy(shares, reason="Signal"):
    """Execute sell order"""
    if check_halt():
        return False

    price = float(rh.stocks.get_latest_price(TICKER)[0])

    pos = get_current_position()
    if not pos or pos["shares"] < shares:
        available = pos["shares"] if pos else 0
        print(f"SELL BLOCKED: Only {available} shares available")
        return False

    try:
        order = rh.orders.order_sell_market(TICKER, shares)
        status = order.get("state", "unknown")
        print(f"SELL ORDER: {shares} {TICKER} @ ~${price:.2f} - Status: {status}")
        log_trade("SELL", TICKER, shares, price, reason, "executed")
        notify(f"SELL EXECUTED: {shares} {TICKER} @ ${price:.2f} - {reason}")
        update_positions()
        return True
    except Exception as e:
        print(f"SELL FAILED: {e}")
        log_trade("SELL", TICKER, shares, price, reason, f"failed: {e}")
        notify(f"SELL FAILED: {shares} {TICKER} - {e}")
        return False'''

new_sell = '''def sell_spy(shares, reason="Signal"):
    """Execute sell order with realized P&L tracking"""
    if check_halt():
        return False

    price = float(rh.stocks.get_latest_price(TICKER)[0])

    pos = get_current_position()
    if not pos or pos["shares"] < shares:
        available = pos["shares"] if pos else 0
        print(f"SELL BLOCKED: Only {available} shares available")
        return False

    avg_cost = pos["avg_cost"]

    try:
        order = rh.orders.order_sell_market(TICKER, shares)
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        print(f"SELL ORDER: {shares} {TICKER} @ ~${price:.2f} - Status: {status}")
        log_trade("SELL", TICKER, shares, price, reason, status)

        # Log realized P&L
        if status == "filled":
            rpnl = log_realized_pnl(price, avg_cost, shares, TICKER)
            notify(f"SELL {status.upper()}: {shares} {TICKER} @ ${price:.2f} (P&L: ${rpnl:.2f}) - {reason}")
        else:
            notify(f"SELL {status.upper()}: {shares} {TICKER} @ ${price:.2f} - {reason}")

        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"SELL FAILED: {e}")
        log_trade("SELL", TICKER, shares, price, reason, f"failed: {e}")
        notify(f"SELL FAILED: {shares} {TICKER} - {e}")
        return False'''

content = content.replace(old_sell, new_sell)

# Update sell_spy_dollars with realized P&L logging and order polling
old_sell_d = '''def sell_spy_dollars(amount, reason="Signal"):
    """Execute sell order by dollar amount (fractional shares)"""
    if check_halt():
        return False

    pos = get_current_position()
    if not pos or pos["equity"] <= 0:
        print("SELL BLOCKED: No position to sell")
        return False

    if amount > pos["equity"]:
        print(f"SELL BLOCKED: ${amount} exceeds position equity ${pos['equity']:.2f}")
        return False

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_sell_fractional_by_price(TICKER, amount, timeInForce='gfd')
        status = order.get("state", "unknown")
        est_shares = round(amount / price, 4)
        print(f"SELL ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("SELL", TICKER, est_shares, price, reason, "executed")
        notify(f"SELL EXECUTED: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")
        update_positions()
        return True
    except Exception as e:
        print(f"SELL FAILED: {e}")
        price = float(rh.stocks.get_latest_price(TICKER)[0]) if True else 0
        log_trade("SELL", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"SELL FAILED: ${amount} {TICKER} - {e}")
        return False'''

new_sell_d = '''def sell_spy_dollars(amount, reason="Signal"):
    """Execute sell order by dollar amount (fractional shares) with realized P&L"""
    if check_halt():
        return False

    pos = get_current_position()
    if not pos or pos["equity"] <= 0:
        print("SELL BLOCKED: No position to sell")
        return False

    if amount > pos["equity"]:
        print(f"SELL BLOCKED: ${amount} exceeds position equity ${pos['equity']:.2f}")
        return False

    avg_cost = pos["avg_cost"]

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_sell_fractional_by_price(TICKER, amount, timeInForce='gfd')
        order_id = order.get("id", None)
        status = poll_order_status(order_id)
        est_shares = round(amount / price, 4)
        print(f"SELL ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("SELL", TICKER, est_shares, price, reason, status)

        # Log realized P&L
        if status == "filled":
            rpnl = log_realized_pnl(price, avg_cost, est_shares, TICKER)
            notify(f"SELL {status.upper()}: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}, P&L: ${rpnl:.2f}) - {reason}")
        else:
            notify(f"SELL {status.upper()}: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")

        update_positions()
        return status == "filled"
    except Exception as e:
        print(f"SELL FAILED: {e}")
        try:
            price = float(rh.stocks.get_latest_price(TICKER)[0])
        except:
            price = 0
        log_trade("SELL", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"SELL FAILED: ${amount} {TICKER} - {e}")
        return False'''

content = content.replace(old_sell_d, new_sell_d)

# Update status_report to show realized P&L
old_status_pnl = '    report.append(f"Daily P&L: ${daily_pnl:.2f} (limit: -${MAX_DAILY_LOSS})")'
new_status_pnl = '    report.append(f"Daily P&L (realized): ${daily_pnl:.2f} (limit: -${MAX_DAILY_LOSS})")'
content = content.replace(old_status_pnl, new_status_pnl)

# Write patched file
with open(filepath, "w") as f:
    f.write(content)

print("✓ spy_trader.py patched with all 4 fixes")
print("")
print("Fixes applied:")
print("  1. P&L now tracks ACTUAL realized profit/loss (not gross cash flow)")
print("     - New file: realized_pnl.csv logs every closed trade's real P&L")
print("     - Daily loss limit only triggers on actual losses")
print("  2. skip_loss_check flag prevents sell+rebuy from blocking itself")
print("  3. Order polling: waits up to 15 sec for fill confirmation")
print("     - Status shows 'filled', 'cancelled', 'rejected', or 'timeout'")
print("  4. All ntfy messages logged to ntfy_log.txt with timestamps")
print("")
print("Reset today's bad data:")
print("  rm executed_trades.csv   # Clear bogus trade log from today")
print("  # realized_pnl.csv will be created fresh on next sell")
print("  # ntfy_log.txt will be created on next notification")
