#!/usr/bin/env python3
"""
Patch spy_trader.py to add fractional/dollar-amount trading.
Run: python3 patch_fractional.py
"""

import re

filepath = "/home/mm/moltbot-research/spy_trader.py"

with open(filepath, "r") as f:
    content = f.read()

# === PATCH 1: Add can_buy_dollars function after can_buy ===
can_buy_dollars_func = '''
def can_buy_dollars(amount):
    """Check all risk controls before buying by dollar amount"""
    reasons = []

    if check_halt():
        return False, "HALTED"

    if get_todays_trades() >= MAX_DAILY_TRADES:
        reasons.append(f"Daily trade limit reached ({MAX_DAILY_TRADES})")

    daily_pnl = get_todays_pnl()
    if daily_pnl <= -MAX_DAILY_LOSS:
        reasons.append(f"Daily loss limit hit (${daily_pnl:.2f})")

    if amount > MAX_POSITION_DOLLARS:
        reasons.append(f"Order ${amount:.2f} exceeds max position ${MAX_POSITION_DOLLARS}")

    cash = get_cash()
    if amount > cash:
        reasons.append(f"Insufficient cash: ${cash:.2f} < ${amount:.2f}")

    pos = get_current_position()
    if pos:
        total_exposure = pos["equity"] + amount
        if total_exposure > MAX_POSITION_DOLLARS:
            reasons.append(f"Total exposure ${total_exposure:.2f} exceeds max ${MAX_POSITION_DOLLARS}")

    if reasons:
        return False, "; ".join(reasons)
    return True, "All checks passed"

'''

# Insert after can_buy function (before check_stop_loss)
content = content.replace(
    'def check_stop_loss():',
    can_buy_dollars_func + 'def check_stop_loss():'
)

# === PATCH 2: Add buy_spy_dollars function after buy_spy ===
buy_dollars_func = '''
def buy_spy_dollars(amount, reason="Signal"):
    """Execute buy order by dollar amount (fractional shares)"""
    can, msg = can_buy_dollars(amount)

    if not can:
        print(f"BUY BLOCKED: {msg}")
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"blocked: {msg}")
        notify(f"BUY BLOCKED: ${amount} {TICKER} - {msg}")
        return False

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_buy_fractional_by_price(TICKER, amount, timeInForce='gfd')
        status = order.get("state", "unknown")
        est_shares = round(amount / price, 4)
        print(f"BUY ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("BUY", TICKER, est_shares, price, reason, "executed")
        notify(f"BUY EXECUTED: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")
        update_positions()
        return True
    except Exception as e:
        print(f"BUY FAILED: {e}")
        price = float(rh.stocks.get_latest_price(TICKER)[0]) if True else 0
        log_trade("BUY", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"BUY FAILED: ${amount} {TICKER} - {e}")
        return False

'''

content = content.replace(
    'def sell_spy(',
    buy_dollars_func + 'def sell_spy('
)

# === PATCH 3: Add sell_spy_dollars function after sell_spy ===
sell_dollars_func = '''
def sell_spy_dollars(amount, reason="Signal"):
    """Execute sell order by dollar amount (fractional shares)"""
    if check_halt():
        return False

    pos = get_current_position()
    if not pos or pos["equity"] <= 0:
        print("SELL BLOCKED: No position to sell")
        return False

    if amount > pos["equity"]:
        print(f"SELL BLOCKED: ${amount} exceeds position equity ${pos['equity']:.2f}")
        return False

    try:
        price = float(rh.stocks.get_latest_price(TICKER)[0])
        order = rh.orders.order_sell_fractional_by_price(TICKER, amount, timeInForce='gfd')
        status = order.get("state", "unknown")
        est_shares = round(amount / price, 4)
        print(f"SELL ORDER: ${amount} {TICKER} (~{est_shares} shares @ ~${price:.2f}) - Status: {status}")
        log_trade("SELL", TICKER, est_shares, price, reason, "executed")
        notify(f"SELL EXECUTED: ${amount} {TICKER} (~{est_shares} shares @ ${price:.2f}) - {reason}")
        update_positions()
        return True
    except Exception as e:
        print(f"SELL FAILED: {e}")
        price = float(rh.stocks.get_latest_price(TICKER)[0]) if True else 0
        log_trade("SELL", TICKER, f"${amount}", price, reason, f"failed: {e}")
        notify(f"SELL FAILED: ${amount} {TICKER} - {e}")
        return False

'''

content = content.replace(
    'def sell_all(',
    sell_dollars_func + 'def sell_all('
)

# === PATCH 4: Update sell_all to handle fractional shares ===
# Change int(pos["shares"]) to float for fractional
content = content.replace(
    'return sell_spy(int(pos["shares"]), reason)',
    'shares = pos["shares"]\n    if shares != int(shares):\n        # Fractional position - sell by dollar amount\n        return sell_spy_dollars(pos["equity"], reason)\n    return sell_spy(int(shares), reason)'
)

# === PATCH 5: Update CLI to support new commands ===
old_cli = '''        else:
            print("Commands: status, buy <shares>, sell <shares>, sell_all, positions, halt, resume")'''

new_cli = '''        elif cmd == "buy_dollars" and len(sys.argv) > 2:
            buy_spy_dollars(float(sys.argv[2]), "Manual buy (dollars)")
        elif cmd == "sell_dollars" and len(sys.argv) > 2:
            sell_spy_dollars(float(sys.argv[2]), "Manual sell (dollars)")
        else:
            print("Commands: status, buy <shares>, sell <shares>, buy_dollars <amount>, sell_dollars <amount>, sell_all, positions, halt, resume")'''

content = content.replace(old_cli, new_cli)

# Write patched file
with open(filepath, "w") as f:
    f.write(content)

print("✓ spy_trader.py patched successfully")
print("")
print("New commands:")
print("  python3 spy_trader.py buy_dollars 100    # Buy $100 worth of SPY")
print("  python3 spy_trader.py sell_dollars 50     # Sell $50 worth of SPY")
print("  python3 spy_trader.py sell_all            # Now handles fractional shares")
print("")
print("Moltbot can now be told:")
print('  "Buy $100 worth of SPY via spy_trader.py buy_dollars 100"')
