#!/usr/bin/env python3
"""
SPY Signal Generator - Moltbot Integration
Analyzes sentiment from financial sources and generates trade signals.
Moltbot calls this or follows these rules manually via web research.
"""

import os
import json
from datetime import datetime

WORKSPACE = "/home/mm/moltbot-research"
SIGNALS_FILE = os.path.join(WORKSPACE, "signals.md")
PENDING_FILE = os.path.join(WORKSPACE, "pending_trades.md")

# ============================================
# SIGNAL RULES
# These are the rules Moltbot follows when
# doing web research and generating signals.
# ============================================

SIGNAL_RULES = """
## SPY TRADING SIGNAL RULES

### Data Sources (check in order)
1. Yahoo Finance - SPY price, volume, premarket
2. CNBC / Reuters / Bloomberg - market headlines
3. StockTwits - SPY sentiment feed
4. Reddit r/wallstreetbets, r/stocks - SPY mentions
5. X/Twitter - $SPY hashtag, financial accounts
6. VIX level (fear gauge)

### BUY Signal (all must be true)
- No current position OR position < $250
- Market sentiment is BULLISH (more positive than negative headlines)
- SPY is not at all-time high (within last 5 trading days)
- VIX is below 25 (market not in panic)
- No major negative catalyst in last 2 hours (Fed announcement, geopolitical shock)
- Volume is normal or above average

### SELL Signal (any one triggers)
- Position P&L hits -3% (stop loss - AUTOMATIC via risk controls)
- Position P&L hits +5% (take profit - AUTOMATIC via risk controls)
- Sentiment shifts strongly BEARISH (3+ major sources negative)
- VIX spikes above 30
- Major negative news (war escalation, bank failure, surprise Fed action)

### HOLD (do nothing when)
- Mixed signals (some bullish, some bearish)
- Low volume / premarket only
- No clear catalyst in either direction
- Position is profitable but below take-profit target

### Signal Scoring
Rate each source: BULLISH (+1), NEUTRAL (0), BEARISH (-1)

| Source | Score |
|--------|-------|
| Yahoo Finance trend | |
| Major headlines | |
| StockTwits sentiment | |
| Reddit sentiment | |
| X/Twitter sentiment | |
| VIX level | |
| TOTAL | |

Score >= +2: BUY signal
Score <= -3: SELL signal
Score -2 to +2: HOLD

### Position Sizing
- First buy: 1 share (~$580-600)
- If signal strengthens: add 1 share (max $250 total)
- Never exceed MAX_POSITION_DOLLARS ($250) - risk controls enforce this
- Note: At current SPY price, $250 limit means ~0-1 shares max

### Timing
- Check signals at market open (9:30 AM ET / 6:30 AM AZ)
- Recheck at midday (12:00 PM ET / 9:00 AM AZ)  
- Recheck at 2:00 PM ET / 11:00 AM AZ
- No trades in last 15 min before close (3:45-4:00 PM ET)
"""

def write_signal_report(scores, analysis, recommendation):
    """Write signal analysis to signals.md"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M")
    
    report = f"""# SPY Signal Report - {timestamp}

## Sentiment Scores

| Source | Score | Notes |
|--------|-------|-------|
"""
    for source, data in scores.items():
        report += f"| {source} | {data['score']} | {data['notes']} |\n"
    
    total = sum(d['score'] for d in scores.values())
    report += f"| **TOTAL** | **{total}** | |\n"
    
    report += f"""
## Analysis
{analysis}

## Recommendation
**{recommendation}**

Total Score: {total}
- Score >= +2: BUY
- Score <= -3: SELL  
- Score -2 to +2: HOLD

---
"""
    
    # Append to signals file
    with open(SIGNALS_FILE, "a") as f:
        f.write(report)
    
    print(f"Signal report written. Score: {total}, Rec: {recommendation}")
    return total, recommendation


def write_pending_trade(action, shares, price, reason):
    """Write proposed trade to pending_trades.md for review"""
    timestamp = datetime.now().strftime("%Y-%m-%d %H:%M")
    
    entry = f"""## Pending Trade - {timestamp}
- **Action:** {action}
- **Ticker:** SPY
- **Shares:** {shares}
- **Approx Price:** ${price:.2f}
- **Reason:** {reason}
- **Status:** AWAITING APPROVAL

---
"""
    with open(PENDING_FILE, "a") as f:
        f.write(entry)
    
    print(f"Pending trade written: {action} {shares} SPY")


# Print rules if run directly
if __name__ == "__main__":
    print(SIGNAL_RULES)
